<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class sbv_validator {
	public $products = array();

	public function __construct() {
	}

	public static function instance() : sbv_validator {
		static $instance = null;

		if ( ! isset( $instance ) ) {
			$instance = new sbv_validator();
		}

		return $instance;
	}

	public function get( $name ) {
		return smart_sbv_core()->get( $name );
	}

	public function validate_item( $purchase_code, $item_id, $api_username = '', $api_code = '' ) : array {
		if ( $api_code == '' && $api_username == '' ) {
			$api_username = $this->get( 'global_user_name' );
			$api_code     = $this->get( 'global_api_token' );
		}

		$found      = false;
		$error      = false;
		$error_code = false;

		if ( $api_code != '' && $api_username != '' ) {
			sbv_api()->data( 10 );
			sbv_api()->access_custom( $api_username, $api_code );

			$verification = sbv_api()->verify_purchase( $purchase_code );

			if ( is_wp_error( $verification ) ) {
				$error      = $verification->get_error_message();
				$error_code = $verification->get_error_code();

				if ( $error_code == 'curl_error' ) {
					$error = __( 'Purchase code verification failed due to connection problems. Please try again later.', 'smart-bbpress-nverify' ) . '<br />' . $error;
				}

				if ( $error_code == 'envato_error' ) {
					$error = __( 'Purchase code verification failed due to the problem with Envato API. Please try again later.', 'smart-bbpress-nverify' );
				}
			} else if ( is_object( $verification ) && ! isset( $verification->item_id ) ) {
				$error      = __( 'Purchase code is not valid.', 'smart-bbpress-nverify' );
				$error_code = 'invalid_code';
			} else if ( $verification->item_id == $item_id ) {
				$found = $verification;
			} else {
				$error = __( 'Purchase code is not valid for this forum.', 'smart-bbpress-nverify' );
			}
		}

		return compact( 'found', 'error', 'error_code' );
	}

	public function validate_account( $purchase_code, $api_username = '', $api_code = '' ) : array {
		if ( $api_code == '' && $api_username == '' ) {
			$api_username = $this->get( 'global_user_name' );
			$api_code     = $this->get( 'global_api_token' );
		}

		$found      = false;
		$error      = false;
		$error_code = false;

		if ( $api_code != '' && $api_username != '' ) {
			sbv_api()->data( 10 );
			sbv_api()->access_custom( $api_username, $api_code );

			$verification = sbv_api()->verify_purchase( $purchase_code );

			if ( is_wp_error( $verification ) ) {
				$error      = $verification->get_error_message();
				$error_code = $verification->get_error_code();

				if ( $error_code == 'curl_error' ) {
					$error = __( 'Purchase code verification failed due to connection problems. Please try again later.', 'smart-bbpress-nverify' ) . '<br />' . $error;
				}

				if ( $error_code == 'envato_error' ) {
					$error = __( 'Purchase code verification failed due to the problem with Envato API. Please try again later.', 'smart-bbpress-nverify' );
				}
			} else if ( is_object( $verification ) && ! isset( $verification->item_id ) ) {
				$error      = __( 'Purchase code is not valid.', 'smart-bbpress-nverify' );
				$error_code = 'invalid_code';
			} else {
				$found = $verification;
			}
		}

		return compact( 'found', 'error', 'error_code' );
	}

	public function revalidate( $user_id = 0, $period = 0 ) {
		$user_id = $user_id == 0 ? get_current_user_id() : $user_id;
		$period  = $period == 0 ? $this->get( 'revalidate_period_days' ) * DAY_IN_SECONDS : $period;

		$current   = time();
		$purchases = sbv_user_list_envato_purchases( $user_id );

		foreach ( $purchases as $item => $codes ) {
			foreach ( $codes as $code => $data ) {
				$last_checked = $data['last_checked'];
				$support_info = isset( $data['support'] );

				if ( ! $support_info || $current > $last_checked + $period ) {
					$this->revalidate_single( $user_id, $item, $code );
				}
			}
		}

		wp_cache_delete( $user_id, 'user_meta' );
	}

	public function revalidate_single( $user_id, $item, $purchase_code ) {
		$this->load_products();

		$current   = time();
		$purchases = sbv_user_list_envato_purchases( $user_id );

		if ( isset( $this->products[ $item ] ) ) {
			$_username = $this->products[ $item ]['username'];
			$_apikey   = $this->products[ $item ]['api_key'];

			if ( $_apikey != '' && $_username != '' ) {
				sbv_api()->data( 10 );
				sbv_api()->access_custom( $_username, $_apikey );

				$verification = sbv_api()->verify_purchase( $purchase_code );

				if ( ! is_wp_error( $verification ) && ! is_null( $verification ) ) {
					if ( is_object( $verification ) && ! isset( $verification->item_id ) ) {
						if ( isset( $purchases[ $item ][ $purchase_code ] ) ) {
							unset( $purchases[ $item ][ $purchase_code ] );

							if ( empty( $purchases[ $item ] ) ) {
								unset( $purchases[ $item ] );
							}
						}

						delete_user_meta( $user_id, 'sbv_envato_' . $item, $purchase_code );

						do_action( 'sbv_purchase_code_revalidation_missing', $user_id, $purchase_code, $verification );

						if ( empty( $purchases ) ) {
							delete_user_meta( $user_id, 'sbv_envato_purchases' );
						} else {
							update_user_meta( $user_id, 'sbv_envato_purchases', $purchases );
						}
					} else {
						$_item = $purchases[ $item ][ $purchase_code ];

						if ( isset( $_item['missing'] ) ) {
							unset( $_item['missing'] );
						}

						$_item['data']         = (array) $verification;
						$_item['last_checked'] = $current;
						$_item['support']      = array(
							'active'    => true,
							'until'     => null,
							'september' => false,
						);

						$purchase_date  = strtotime( $verification->created_at );
						$september_date = strtotime( '1 September 2015' );

						if ( $purchase_date < $september_date ) {
							$_item['support']['september'] = true;
						}

						if ( empty( $verification->supported_until ) ) {
							$_item['support']['active'] = false;
						} else {
							$until = strtotime( $verification->supported_until ) + DAY_IN_SECONDS;

							if ( time() < $until ) {
								$_item['support']['until'] = $until;
							} else {
								$_item['support']['active'] = false;
							}
						}

						$purchases[ $item ][ $purchase_code ] = $_item;

						do_action( 'sbv_purchase_code_revalidated', $user_id, $purchase_code, $_item );

						update_user_meta( $user_id, 'sbv_envato_purchases', $purchases );
					}
				}
			}
		} else {
			$_item = $purchases[ $item ][ $purchase_code ];

			$_item['missing'] = true;

			$expired = true;

			if ( ! isset( $item['support'] ) ) {
				$_item['support'] = array(
					'active'    => false,
					'until'     => null,
					'september' => false,
				);
			} else {
				if ( ! is_null( $_item['support']['until'] ) ) {
					$until   = intval( $_item['support']['until'] );
					$expired = $until < time();
				}
			}

			if ( $expired ) {
				$_item['support']['active'] = $this->get( 'global_support_retired' );
			}

			if ( isset( $_item['data']['created_at'] ) ) {
				$purchase_date  = strtotime( $_item['data']['created_at'] );
				$september_date = strtotime( '1 September 2015' );

				if ( $purchase_date < $september_date ) {
					$_item['support']['september'] = true;
				}
			}

			$purchases[ $item ][ $purchase_code ] = $_item;

			update_user_meta( $user_id, 'sbv_envato_purchases', $purchases );
		}
	}

	public function load_products() {
		if ( empty( $this->products ) ) {
			global $wpdb;

			$query = "select post_id, meta_value from " . $wpdb->postmeta . " where meta_key = 'sbn_envato_items' and meta_value != ''";
			$rows  = $wpdb->get_results( $query );

			foreach ( $rows as $row ) {
				$raw = maybe_unserialize( $row->meta_value );

				foreach ( $raw['items'] as $item ) {
					if ( ! isset( $this->products[ $item['item'] ] ) ) {
						$item['forum'] = $row->post_id;

						if ( $item['author'] != 'custom' ) {
							$item['username'] = $this->get( 'global_user_name' );
							$item['api_key']  = $this->get( 'global_api_token' );
						}

						$this->products[ $item['item'] ] = $item;
					}
				}
			}
		}
	}

	public function is_purchase_code_unique( $purchase_code ) : bool {
		if ( $this->get( 'verify_unique_purchase_code' ) ) {
			global $wpdb;

			$query = "SELECT COUNT(*) FROM " . $wpdb->usermeta . " WHERE `meta_value` = '" . esc_sql( $purchase_code ) . "' and `meta_key` LIKE 'sbv_envato_%'";
			$count = $wpdb->get_var( $query );

			return $count == 0;
		} else {
			return true;
		}
	}

	public function add_purchase_code_to_user( $purchase_code, $user_id, $item_id = 0, $api_username = '', $api_code = '' ) {
		$found      = false;
		$error      = false;
		$error_code = false;

		$unique = sbv_validator()->is_purchase_code_unique( $purchase_code );

		if ( $unique ) {
			if ( $item_id == 0 ) {
				$result = $this->validate_account( $purchase_code, $api_username, $api_code );
			} else {
				$result = $this->validate_item( $purchase_code, $item_id, $api_username, $api_code );
			}

			extract( $result );
		} else {
			$error      = __( 'Purchase code is already registered by another user.', 'smart-bbpress-nverify' );
			$error_code = 'already_registered';
		}

		if ( $found !== false ) {
			sbv_store_purchase_code( $user_id, $purchase_code, $found );

			return true;
		} else {
			return new WP_Error( $error_code, $error );
		}
	}
}
