<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class sbv_bbpress {
	var $support_check = 'ok';
	var $support_expired = array();

	var $bulk_store_errors = false;

	function __construct() {
		add_action( 'bbp_init', array( $this, 'init' ) );

		add_action( 'init', array( $this, 'bulk' ) );
		add_action( 'init', array( $this, 'redirect' ) );

		add_filter( 'bbp_request', array( $this, 'request_feed_trap' ), 1 );
	}

	public static function instance() : sbv_bbpress {
		static $instance = null;

		if ( ! isset( $instance ) ) {
			$instance = new sbv_bbpress();
		}

		return $instance;
	}

	public function request_feed_trap( $query_vars = array() ) {
		if ( isset( $query_vars['feed'] ) && $this->get( 'integrate_bbpress_disable_rss_feeds' ) ) {
			if ( isset( $query_vars['post_type'] ) ) {
				$post_type  = false;
				$post_types = array(
					bbp_get_forum_post_type(),
					bbp_get_topic_post_type(),
					bbp_get_reply_post_type(),
				);

				$qv_array = (array) $query_vars['post_type'];

				foreach ( $post_types as $bbp_pt ) {
					if ( in_array( $bbp_pt, $qv_array, true ) ) {
						$post_type = $bbp_pt;
						break;
					}
				}

				if ( ! empty( $post_type ) ) {
					switch ( $post_type ) {
						case bbp_get_forum_post_type():
							if ( isset( $query_vars['name'] ) && ! empty( $query_vars['name'] ) ) {
								$page = get_page_by_path( $query_vars['name'], OBJECT, bbp_get_forum_post_type() );

								if ( is_array( $page ) ) {
									$page = $page[0];
								}

								if ( $page && isset( $page->ID ) ) {
									wp_safe_redirect( get_permalink( $page->ID ) );
								} else {
									wp_safe_redirect( bbp_get_forums_url() );
								}
							} else {
								wp_safe_redirect( bbp_get_forums_url() );
							}

							exit;
						case bbp_get_topic_post_type():
							if ( isset( $query_vars['name'] ) && ! empty( $query_vars['name'] ) ) {
								$page = $this->get_topic_id_from_slug( $query_vars['name'] );

								if ( $page ) {
									wp_safe_redirect( get_permalink( $page ) );
								} else {
									wp_safe_redirect( bbp_get_forums_url() );
								}
							} else {
								wp_safe_redirect( bbp_get_forums_url() );
							}

							exit;
						case bbp_get_reply_post_type():
							wp_safe_redirect( bbp_get_forums_url() );
							exit;
					}
				}
			} else if ( isset( $query_vars[ bbp_get_view_rewrite_id() ] ) ) {
				$view = $query_vars[ bbp_get_view_rewrite_id() ];
				wp_safe_redirect( bbp_get_view_url( $view ) );
			}
		}

		return $query_vars;
	}

	private function get_topic_id_from_slug( $slug ) {
		global $wpdb;

		$slug = esc_sql( $slug );
		$slug = sanitize_title_for_query( $slug );

		$post_type = bbp_get_topic_post_type();

		$sql = $wpdb->prepare( "SELECT ID FROM " . $wpdb->posts . " WHERE post_name = %s AND post_type = %s", $slug, $post_type );

		$pages = $wpdb->get_results( $sql );

		if ( count( $pages ) == 1 ) {
			return $pages[0]->ID;
		}

		return null;
	}

	public function get( $name ) {
		return smart_sbv_core()->get( $name );
	}

	public function ping() {
		if ( $this->get( 'ping_envato_api' ) ) {
			return sbv_api()->is_online();
		} else {
			return true;
		}
	}

	public function bulk() {
		if ( isset( $_POST['sbv'] ) && isset( $_POST['sbv']['bulk'] ) ) {
			if ( $_POST['sbv']['bulk'] == 'add' ) {
				$test = wp_verify_nonce( $_POST['sbv']['verify'], 'smart-bbpress-nverify-' . get_current_user_id() );

				$added  = 0;
				$errors = new WP_Error();

				if ( $test === false ) {
					$errors->add( 'request', __( 'Invalid Request', 'smart-bbpress-nverify' ) );
				} else {
					$user_id = intval( $_POST['sbv']['user'] );
					$codes   = (array) $_POST['sbv']['codes'];

					$codes = array_map( 'trim', $codes );
					$codes = array_filter( $codes );

					foreach ( $codes as $purchase_code ) {
						$found      = false;
						$error_code = false;

						$unique = sbv_validator()->is_purchase_code_unique( $purchase_code );

						if ( $unique ) {
							$result = sbv_validator()->validate_account( $purchase_code );
							extract( $result );

							if ( $error_code === 'invalid_code' ) {
								$errors->add( 'invalid_code', sprintf( __( 'Purchase code \'%s\' is not valid.', 'smart-bbpress-nverify' ), $purchase_code ) );
							} else if ( $error_code !== false ) {
								$errors->add( 'error', sprintf( __( 'Verification failed for code \'%s\', please try again later.', 'smart-bbpress-nverify' ), $purchase_code ) );
							}

							if ( $found !== false ) {
								do_action( 'sbv_purchase_code_validation_failed_bulk', $user_id, $purchase_code, $result );
							}
						} else {
							$errors->add( 'invalid_code', sprintf( __( 'Purchase code \'%s\' is already registered.', 'smart-bbpress-nverify' ), $purchase_code ) );

							do_action( 'sbv_purchase_code_already_registered_bulk', $user_id, $purchase_code );
						}

						if ( $found !== false ) {
							sbv_store_purchase_code( $user_id, $purchase_code, $found );

							$added ++;
						}
					}
				}

				if ( empty( $errors->errors ) && $added == 0 ) {
					$errors->add( 'error', __( 'No valid codes submitted.', 'smart-bbpress-nverify' ) );
				}

				if ( empty( $errors->errors ) ) {
					$url = add_query_arg( 'sbvbulk', $added, bbp_get_user_profile_url( $user_id ) );

					wp_redirect( $url . '#sbv-profile-purchase-codes' );
					exit;
				} else {
					$this->bulk_store_errors = $errors;
				}
			}
		}
	}

	public function redirect() {
		if ( $this->get( 'handle_envato_support_urls' ) ) {
			if ( isset( $_GET['envato_item_id'] ) ) {
				$item = $_GET['envato_item_id'];

				sbv_validator()->load_products();

				if ( isset( sbv_validator()->products[ $item ] ) ) {
					wp_redirect( get_permalink( sbv_validator()->products[ $item ]['forum'] ) );
					exit;
				}
			}
		}
	}

	public function check( $type, $forum_id = 0 ) : bool {
		$user_id    = get_current_user_id();
		$revalidate = array();

		$skip = sbv_is_current_user_with_role( $this->get( 'access_unrestriced_user_roles' ) );
		$skip = apply_filters( 'sbv_skip_protect_check_' . $type, $skip, $user_id );

		if ( ! $skip ) {
			$items = sbv_forum_list_required_items( $forum_id );

			$support    = apply_filters( 'sbv_check_support_status', $this->get( 'global_support' ) );
			$september  = apply_filters( 'sbv_check_support_september', $this->get( 'global_support_september' ) );
			$retired    = apply_filters( 'sbv_check_support_retired', $this->get( 'global_support_retired' ) );
			$exceptions = apply_filters( 'sbv_check_support_exceptions', $this->get( 'global_support_exceptions' ) );

			$protect = false;

			if ( $items === 'account' || ! empty( $items ) ) {
				if ( ! is_user_logged_in() ) {
					$protect = true;
				} else {
					if ( $this->get( 'revalidate_purchase_code' ) ) {
						sbv_validator()->revalidate();
					}

					$purchases = sbv_user_list_envato_purchases();

					$protect = true;

					if ( $items === 'account' ) {
						if ( ! empty( $purchases ) ) {
							$protect = false;
						}
					} else {
						foreach ( $items as $item ) {
							if ( isset( $purchases[ $item->id ] ) ) {
								if ( $support && ! in_array( $item->id, $exceptions ) && $item->sbv_support == 'auto' ) {
									$supported = - 1;

									foreach ( $purchases[ $item->id ] as $code => $purchase ) {
										if ( isset( $purchase['support'] ) ) {
											$status = $purchase['support']['active'];

											if ( $september && isset( $purchase['support']['september'] ) && $purchase['support']['september'] ) {
												$status = true;
											}

											if ( $retired && isset( $purchase['missing'] ) && $purchase['missing'] ) {
												$status = true;
											}

											if ( $supported === - 1 ) {
												$supported = $status;
											} else if ( $supported === false ) {
												$supported = $status;
											}

											if ( ! $status ) {
												$last_checked = $purchase['last_checked'];

												if ( time() > $last_checked + MINUTE_IN_SECONDS * 5 ) {
													$revalidate[] = array(
														'user' => $user_id,
														'code' => $code,
														'item' => $item->id,
													);
												}
											}
										}
									}

									if ( $supported === - 1 ) {
										$supported = true;
									}

									$protect = ! $supported;

									if ( ! $supported ) {
										$this->support_check     = 'support_expired';
										$this->support_expired[] = $item->id;
									}
								} else {
									$protect = false;
								}

								if ( ! $protect ) {
									break;
								}
							}
						}
					}
				}
			}

			foreach ( $revalidate as $rev ) {
				sbv_validator()->revalidate_single( $rev['user'], $rev['item'], $rev['code'] );
			}

			return $protect;
		} else {
			return false;
		}
	}

	public function init() {
		add_action( 'bbp_enqueue_scripts', array( $this, 'bbp_enqueue_scripts' ) );
		add_filter( 'bbp_get_template_part', array( $this, 'bbp_get_template_part' ), 10, 3 );
		add_filter( 'bbp_add_template_stack_locations', array( $this, 'bbp_add_template_stack_locations' ) );

		if ( $this->get( 'access_control_new_topic_noforum' ) ) {
			add_action( 'bbp_new_topic_pre_extras', array( $this, 'bbp_new_topic_pre_extras' ) );
			add_action( 'bbp_template_notices', array( $this, 'bbp_template_notices' ), 1000 );
		}

		if ( $this->get( 'integrate_bbpress_profile_active' ) ) {
			add_action( 'bbp_template_after_user_profile', array( $this, 'bbp_template_after_user_profile' ) );
		}

		do_action( 'sbv_init_bbpress' );
	}

	public function protect( $templates, $name, $type, $forum_id = 0 ) {
		$protect = $this->check( $type, $forum_id );

		if ( $protect ) {
			array_unshift( $templates, 'nverify-' . $name . '.php' );
		}

		return $templates;
	}

	public function bbp_template_after_user_profile() {
		$view = $bulk = bbp_is_user_home();

		if ( sbv_is_current_user_with_role() ) {
			if ( $this->get( 'integrate_bbpress_profile_visible_to_administrators' ) ) {
				$view = true;
			}

			if ( $this->get( 'integrate_bbpress_profile_bulk_visible_to_administrators' ) ) {
				$bulk = true;
			}
		}

		if ( sbv_is_current_user_with_role( 'bbp_moderator' ) ) {
			if ( $this->get( 'integrate_bbpress_profile_visible_to_moderators' ) ) {
				$view = true;
			}

			if ( $this->get( 'integrate_bbpress_profile_bulk_visible_to_moderators' ) ) {
				$bulk = true;
			}
		}

		if ( $view ) {
			bbp_get_template_part( 'nverify', 'user-profile-list' );
		}

		if ( $bulk ) {
			bbp_get_template_part( 'nverify', 'user-profile-bulk' );
		}
	}

	public function bbp_new_topic_pre_extras( $forum_id ) {
		if ( $forum_id > 0 ) {
			$protect = $this->check( 'topic-form', $forum_id );

			if ( $protect ) {
				bbp_add_error( 'sbv_purchase_code_missing', __( '<strong>ERROR</strong>: Access to this forum requires valid purchase code.', 'smart-bbpress-nverify' ) );
			}
		}
	}

	public function bbp_template_notices() {
		if ( bbp_has_errors() ) {
			$bbp = bbpress();

			if ( isset( $bbp->errors->errors['sbv_purchase_code_missing'] ) ) {
				global $sbv_forum_id, $sbv_validate_action, $sbv_default_notice;

				$sbv_forum_id        = bbp_get_form_topic_forum();
				$sbv_validate_action = 'hide';

				$templates = $this->protect( array(), 'form-topic', 'topic-form', $sbv_forum_id );

				$load = bbp_locate_template( $templates );

				include( $load );
			}
		}
	}

	public function bbp_get_template_part( $templates, $slug, $name ) {
		if ( $this->get( 'access_control_search' ) && $slug == 'loop' && $name == 'search-' . bbp_get_topic_post_type() ) {
			$_topic_id = bbp_get_topic_id();
			$_forum_id = bbp_get_topic_forum_id( $_topic_id );

			$templates = $this->protect( $templates, 'search-topic', 'topic', $_forum_id );
		}

		if ( $this->get( 'access_control_search' ) && $slug == 'loop' && $name == 'search-' . bbp_get_reply_post_type() ) {
			$_reply_id = bbp_get_reply_id();
			$_forum_id = bbp_get_reply_forum_id( $_reply_id );

			$templates = $this->protect( $templates, 'search-reply', 'reply', $_forum_id );
		}

		if ( $this->get( 'access_control_forum' ) && $slug == 'content' && $name == 'single-' . bbp_get_forum_post_type() ) {
			$templates = $this->protect( $templates, 'single-forum', 'forum' );
		}

		if ( $this->get( 'access_control_topic' ) && $slug == 'content' && $name == 'single-' . bbp_get_topic_post_type() ) {
			$templates = $this->protect( $templates, 'single-topic', 'topic' );
		}

		if ( $this->get( 'access_control_new_topic' ) && $slug == 'form' && $name == bbp_get_topic_post_type() ) {
			$templates = $this->protect( $templates, 'form-topic', 'topic-form' );
		}

		if ( $this->get( 'access_control_new_reply' ) && $slug == 'form' && $name == bbp_get_reply_post_type() ) {
			$templates = $this->protect( $templates, 'form-reply', 'reply-form' );
		}

		if ( $this->get( 'access_control_reply' ) && $slug == 'loop' && $name == 'single-' . bbp_get_reply_post_type() ) {
			$_reply_id = bbp_get_reply_id();
			$_forum_id = bbp_get_reply_forum_id( $_reply_id );

			$templates = $this->protect( $templates, 'single-reply', 'reply', $_forum_id );
		}

		return $templates;
	}

	public function bbp_add_template_stack_locations( $locations ) : array {
		$locations[] = SBV_PATH . 'bbpress/';

		return $locations;
	}

	public function bbp_enqueue_scripts() {
		$is_debug = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG;

		$_load_css = apply_filters( 'sbv_load_css', true );
		$_load_js  = apply_filters( 'sbv_load_js', true );

		if ( $_load_css ) {
			wp_enqueue_style( 'sbv-render-css', ( $is_debug ? SBV_URL . 'css/render.css' : SBV_URL . 'css/render.min.css' ), array(), $this->get( '__version__' ) );
		}

		if ( $_load_js ) {
			wp_enqueue_script( 'jquery' );

			wp_enqueue_script( 'sbv-mask', SBV_URL . 'libs/jquery.maskedinput.min.js', array( 'jquery' ), null, true );
			wp_enqueue_script( 'sbv-render', ( $is_debug ? SBV_URL . 'js/render.js' : SBV_URL . 'js/render.min.js' ), array(
				'jquery',
				'sbv-mask',
			), $this->get( '__version__' ), true );

			wp_localize_script( 'sbv-render', 'sbv_verificaton_data', array(
				'ajax'   => admin_url( 'admin-ajax.php' ),
				'masked' => true,
				'forum'  => bbp_get_forum_id(),
			) );
		}
	}
}
