<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class sbv_customers_grid extends WP_List_Table {
	var $products = array();

	function __construct( $args = array() ) {
		$this->_get_products();

		parent::__construct( array(
			'singular' => 'customer',
			'plural'   => 'customers',
			'ajax'     => false,
		) );
	}

	function rows_per_page() {
		$user     = get_current_user_id();
		$per_page = get_user_meta( $user, 'sbv_rows_per_page_customers', true );

		if ( empty( $per_page ) || $per_page < 1 ) {
			$per_page = 25;
		}

		return $per_page;
	}

	function _get_products() {
		global $wpdb;

		$query = "select meta_value from " . $wpdb->postmeta . " where meta_key = 'sbn_envato_items' and meta_value != ''";
		$rows  = $wpdb->get_results( $query );

		$this->products = array( 0 => esc_html__( 'All Items / Products', 'smart-bbpress-nverify' ) );

		foreach ( $rows as $row ) {
			$raw = maybe_unserialize( $row->meta_value );

			foreach ( $raw['items'] as $item ) {
				if ( ! isset( $this->products[ $item['item'] ] ) ) {
					$envato = sbv_api()->item( $item['item'] );

					if ( ! is_wp_error( $envato ) ) {
						if ( isset( $envato->item ) ) {
							$this->products[ $envato->id ] = $envato->item;
						} else {
							$this->products[ $item['item'] ] = esc_html__( 'ITEM MISSING', 'smart-bbpress-nverify' ) . ': ' . $item['item'];
						}
					}
				}
			}
		}

		asort( $this->products );
	}

	function extra_tablenav( $which ) {
		if ( $which == 'top' ) {
			$_sel_item = $_GET['filter-item'] ?? '';

			echo '<div class="alignleft actions">';
			sbv_draw_dropdown( $this->products, $_sel_item, 'filter-item' );
			submit_button( esc_html__( 'Filter Customers', 'smart-bbpress-nverify' ), 'button', false, false, array( 'id' => 'customer-query-submit' ) );
			echo '</div>';
		}
	}

	function get_columns() {
		return array(
			'id'       => esc_html__( 'User ID', 'smart-bbpress-nverify' ),
			'username' => esc_html__( 'User', 'smart-bbpress-nverify' ),
			'info'     => esc_html__( 'Information', 'smart-bbpress-nverify' ),
			'date'     => esc_html__( 'Registered', 'smart-bbpress-nverify' ),
			'data'     => esc_html__( 'Purchase Codes', 'smart-bbpress-nverify' ),
			'topics'   => esc_html__( 'Posts', 'smart-bbpress-nverify' ),
		);
	}

	function get_sortable_columns() {
		return array(
			'id'       => array( 'ID', false ),
			'username' => array( 'name', false ),
		);
	}

	function column_id( $item ) {
		return $item->ID;
	}

	function column_username( $item ) {
		$display   = $item->display_name != '' ? $item->display_name : $item->user_login;
		$edit_link = esc_url( add_query_arg( 'wp_http_referer', urlencode( wp_unslash( $_SERVER['REQUEST_URI'] ) ), get_edit_user_link( $item->ID ) ) );
		$edit      = "<strong><a href=\"$edit_link\">$display</a></strong><br />";

		$actions = array(
			'edit'       => '<a href="' . $edit_link . '">' . esc_html__( 'Edit', 'smart-bbpress-nverify' ) . '</a>',
			'profile'    => '<a href="' . bbp_get_user_profile_url( $item->ID ) . '">' . esc_html__( 'bbPress Profile', 'smart-bbpress-nverify' ) . '</a>',
			'revalidate' => '<a href="options-general.php?page=smart-bbpress-nverify&tab=customers&single-action=revalidate&user=' . $item->ID . '&_wpnonce=' . wp_create_nonce( 'sbv-revalidate-' . $item->ID ) . '">' . esc_html__( 'Revalidate', 'smart-bbpress-nverify' ) . '</a>',
		);

		return get_avatar( $item->ID, 40 ) . ' ' . $edit . $this->row_actions( $actions );
	}

	function column_topics( $item ) {
		$topics  = $item->data->forums['topic'] ?? 0;
		$replies = $item->data->forums['reply'] ?? 0;

		$render = '';

		if ( $topics == 0 ) {
			$render .= '0 ' . esc_html__( 'Topics', 'smart-bbpress-nverify' );
		} else {
			$render .= '<a href="' . bbp_get_user_topics_created_url( $item->ID ) . '">' . $topics . ' ' . esc_html( _n( 'Topic', 'Topics', $topics, 'smart-bbpress-nverify' ) ) . '</a>';
		}

		if ( $replies == 0 ) {
			$render .= '<br/>0 ' . esc_html__( 'Replies', 'smart-bbpress-nverify' );
		} else {
			$render .= '<br/><a href="' . bbp_get_user_replies_created_url( $item->ID ) . '">' . $replies . ' ' . esc_html( _n( 'Reply', 'Replies', $topics, 'smart-bbpress-nverify' ) ) . '</a>';
		}

		return $render;
	}

	function column_info( $item ) {
		$render = esc_html__( 'Username', 'smart-bbpress-nverify' ) . ': <strong>' . $item->user_login . '</strong><br/>';
		$render .= esc_html__( 'Email', 'smart-bbpress-nverify' ) . ': <strong>' . $item->user_email . '</strong>';

		return $render;
	}

	function column_data( $item ) {
		global $sbv_core_admin;

		$content = array();
		$listing = array();

		foreach ( $item->sbv_envato_purchases as $item_code => $data ) {
			foreach ( $data as $code => $info ) {
				$nonce = wp_create_nonce( 'sbv-remove-' . $item->ID . '-' . $item_code . '-' . $code );

				$delete_url = $sbv_core_admin->admin_page_url . '?page=smart-bbpress-nverify&tab=customers&single-action=unlink&user=' . $item->ID . '&item=' . $item_code . '&code=' . $code . '&_wpnonce=' . $nonce;

				$one_item = '[<a class="sbv-view-code-data" href="#' . $code . '">' . $item_code . '</a>] ' . $info['data']['item_name'];

				if ( isset( $info['missing'] ) ) {
					$one_item .= ' &middot; <strong style="color: #dd00dd;">' . esc_html__( 'item retired', 'smart-bbpress-nverify' ) . '</strong>';
				} else if ( ! isset( $info['support'] ) ) {
					$one_item .= ' &middot; <strong style="color: #00dddd;">' . esc_html__( 'outdated', 'smart-bbpress-nverify' ) . '</strong>';
				} else if ( empty( $info['support']['until'] ) ) {
					$one_item .= ' &middot; <strong style="color: #dd0000;">' . esc_html__( 'no support', 'smart-bbpress-nverify' ) . '</strong>';
				}

				$one_item .= ' [<a data-code="' . $code . '" class="sbv-view-code-remove" href="' . $delete_url . '">' . esc_html__( 'remove', 'smart-bbpress-nverify' ) . '</a>]';

				$listing[] = $one_item;

				$data_info = '<div id="envato-code-' . $code . '">';
				$data_info .= '<table class="widefat"><tbody>';
				$data_info .= '<tr><td>' . esc_html__( 'Purchase code', 'smart-bbpress-nverify' ) . '</td><td>' . $code . '</td></tr>';
				$data_info .= '<tr><td>' . esc_html__( 'Registered', 'smart-bbpress-nverify' ) . '</td><td>' . date( 'r', $info['first_checked'] ) . '</td></tr>';

				if ( isset( $info['support'] ) ) {
					$value     = is_null( $info['support']['until'] ) ? esc_html__( 'Expired', 'smart-bbpress-nverify' ) : date( 'r', $info['support']['until'] );
					$data_info .= '<tr><td>' . esc_html__( 'Support', 'smart-bbpress-nverify' ) . '</td><td>' . $value . '</td></tr>';

					if ( isset( $info['support']['september'] ) ) {
						$old_status = $info['support']['september'] ? esc_html__( 'Yes, purchased before September 1, 2015', 'smart-bbpress-nverify' ) : esc_html__( 'No, purchased after September 1, 2015', 'smart-bbpress-nverify' );
						$data_info  .= '<tr><td>' . esc_html__( 'Old Purchase', 'smart-bbpress-nverify' ) . '</td><td>' . $old_status . '</td></tr>';
					}
				}

				if ( isset( $info['missing'] ) ) {
					$data_info .= '<tr><td>' . esc_html__( 'Retired', 'smart-bbpress-nverify' ) . '</td><td>' . esc_html__( 'Product not available.', 'smart-bbpress-nverify' ) . '</td></tr>';
				}

				$data_info .= '<tr><td style="border-bottom: 1px solid #ddd;">' . esc_html__( 'Last verification', 'smart-bbpress-nverify' ) . '</td><td style="border-bottom: 1px solid #ddd;">' . date( 'r', $info['last_checked'] ) . '</td></tr>';
				foreach ( $info['data'] as $key => $value ) {
					$data_info .= '<tr><td>' . ucfirst( str_replace( '_', ' ', $key ) ) . '</td><td>' . $value . '</td></tr>';
				}
				$data_info .= '</tbody></table></div>';

				$content[] = $data_info;
			}
		}

		return join( '<br/>', $listing ) . '<div style="display: none">' . join( '', $content ) . '</div>';
	}

	function column_date( $item ) {
		return mysql2date( 'Y.m.d', $item->user_registered ) . '<br/>@ ' . mysql2date( 'H:m:s', $item->user_registered );
	}

	function topics_replies() {
		global $wpdb;

		$users = array_keys( $this->items );
		$sql   = "SELECT post_type, post_author, count(*) AS counter FROM " . $wpdb->posts . " WHERE post_type IN ('reply', 'topic') AND post_status IN ('pending', 'publish', 'closed') AND post_author IN (" . join( ', ', $users ) . ") GROUP BY post_type, post_author";
		$raw   = $wpdb->get_results( $sql );

		foreach ( $raw as $row ) {
			$this->items[ $row->post_author ]->data->forums[ $row->post_type ] = $row->counter;
		}
	}

	function prepare_items() {
		$this->_column_headers = array( $this->get_columns(), array(), $this->get_sortable_columns() );

		$per_page = $this->rows_per_page();

		$item    = isset( $_GET['filter-item'] ) && $_GET['filter-item'] != 0 ? esc_sql( $_GET['filter-item'] ) : '';
		$orderby = ! empty( $_GET['orderby'] ) ? esc_sql( $_GET['orderby'] ) : 'ID';
		$order   = ! empty( $_GET['order'] ) ? esc_sql( $_GET['order'] ) : 'desc';
		$search  = ! empty( $_GET['s'] ) ? esc_sql( $_GET['s'] ) : '';

		$paged = ! empty( $_GET['paged'] ) ? esc_sql( $_GET['paged'] ) : '';
		if ( empty( $paged ) || ! is_numeric( $paged ) || $paged <= 0 ) {
			$paged = 1;
		}

		$offset = intval( ( $paged - 1 ) * $per_page );

		$args = array(
			'fields'     => 'all_with_meta',
			'number'     => $per_page,
			'offset'     => $offset,
			'meta_query' => array(
				'relation' => 'AND',
				array(
					'key'     => 'sbv_envato_purchases',
					'value'   => '',
					'compare' => '!=',
				),
			),
		);

		if ( $item != '' && $search == '' ) {
			$args['meta_query'][] = array(
				'key'     => 'sbv_envato_' . $item,
				'value'   => '',
				'compare' => '!=',
			);
		}

		if ( $search != '' ) {
			$args['meta_query'][] = array(
				'key'     => 'sbv_envato_purchases',
				'value'   => $search,
				'compare' => 'LIKE',
			);
		}

		if ( $order != '' ) {
			$args['orderby'] = $orderby;
			$args['order']   = $order;
		}

		$user_query = new WP_User_Query( $args );

		$this->items = $user_query->results;
		$total_items = $user_query->total_users;

		$this->topics_replies();

		$this->set_pagination_args( array(
			'total_items' => $total_items,
			'total_pages' => ceil( $total_items / $per_page ),
			'per_page'    => $per_page,
		) );
	}
}

function sbv_html_id_from_name( $name, $id = '' ) {
	if ( $id == '' ) {
		$id = str_replace( '[', '_', str_replace( ']', '', $name ) );
	} else if ( $id == '_' ) {
		$id = '';
	}

	return $id;
}

function sbv_draw_dropdown_options( $values, $selected, $data ) {
	$render = '';

	foreach ( $values as $value => $display ) {
		$dat = array();

		if ( isset( $data[ $value ] ) ) {
			foreach ( $data[ $value ] as $key => $val ) {
				$dat[] = 'data-' . $key . '="' . $val . '"';
			}
		}

		$sel    = in_array( $value, $selected ) ? ' selected="selected"' : '';
		$render .= '<option value="' . esc_attr( $value ) . '"' . $sel . ' ' . join( ' ', $dat ) . '>' . $display . '</option>';
	}

	return $render;
}

function sbv_draw_dropdown( $values, $selected, $name, $id = '', $attr = array(), $data = array(), $grouped = false, $echo = true ) {
	$render     = '';
	$attributes = array();
	$selected   = (array) $selected;
	$id         = sbv_html_id_from_name( $name, $id );

	foreach ( $attr as $key => $value ) {
		if ( $key == 'multiple' && $value ) {
			$attributes[] = 'multiple';
			$name         .= '[]';
		} else if ( $key == 'size' && $value > 1 ) {
			$attributes[] = 'size="' . $value . '"';
		} else {
			$attributes[] = $key . '="' . $value . '"';
		}
	}

	$attributes[] = 'id="' . $id . '"';
	$attributes[] = 'name="' . $name . '"';

	$render .= '<select ' . join( ' ', $attributes ) . '>';
	if ( $grouped ) {
		foreach ( $values as $group ) {
			$render .= '<optgroup label="' . $group['label'] . '">';
			$render .= sbv_draw_dropdown_options( $group['list'], $selected, $data );
			$render .= '</optgroup>';
		}
	} else {
		$render .= sbv_draw_dropdown_options( $values, $selected, $data );
	}
	$render .= '</select>';

	if ( $echo ) {
		echo $render;
	}

	return $render;
}
