<?php
/**
 * Name:    Smart Envato API: Data Objects
 * Version: 6.1
 * Author:  Milan Petrovic
 * Email:   support@smartplugins.info
 * Website: https://www.smartplugins.info/
 *
 * == Copyright ==
 * Copyright 2008 - 2024 Milan Petrovic (email: support@smartplugins.info)
 */

if ( ! class_exists( 'smart_object_sorting' ) ) {
	class smart_object_sorting {
		var $properties;
		var $sorted;

		function __construct( $objects_array, $properties = array() ) {
			if ( count( $properties ) > 0 ) {
				$this->properties = $properties;
				usort( $objects_array, array( $this, 'array_compare' ) );
			}

			$this->sorted = $objects_array;
		}

		function array_compare( $one, $two, $i = 0 ) : int {
			$column = $this->properties[ $i ]['property'];
			$order  = $this->properties[ $i ]['order'];

			if ( $one->$column == $two->$column ) {
				if ( $i < count( $this->properties ) - 1 ) {
					$i ++;

					return $this->array_compare( $one, $two, $i );
				} else {
					return 0;
				}
			}

			if ( strtolower( $order ) == 'asc' ) {
				return ( $one->$column < $two->$column ) ? - 1 : 1;
			} else {
				return ( $one->$column < $two->$column ) ? 1 : - 1;
			}
		}
	}
}

if ( ! class_exists( 'smart_envato_api_user' ) ) {
	class smart_envato_api_user {
		private $_data = array();
		private $_referrer;
		private $_impact = array(
			'affiliate' => 4415,
			'id'        => 0,
			'ad'        => 0,
		);

		public $badges = null;

		public function __construct( $user, $referrer = '', $badges = null, $impact = array() ) {
			if ( is_array( $user ) || is_object( $user ) ) {
				$this->_data = (array) $user;
			}

			$this->_referrer = $referrer;
			$this->badges    = $badges;

			$this->_impact['affiliate'] = $impact['affiliate'] ?? 4415;
			$this->_impact['id']        = $impact['id'] ?? 0;
			$this->_impact['ad']        = $impact['ad'] ?? 0;
		}

		public function __get( $name ) {
			return $this->_data[ $name ] ?? '';
		}

		public function __set( $name, $value ) {
			$this->_data[ $name ] = $value;
		}

		function __isset( $name ) {
			return isset( $this->_data[ $name ] );
		}

		public function url( $market ) : string {
			$author = 'https://' . strtolower( $market ) . '.net/user/' . $this->username;

			return $this->_affiliate_url( $author );
		}

		public function portfolio_url( $market ) : string {
			$author = 'https://' . strtolower( $market ) . '.net/user/' . $this->username . '/portfolio';

			return $this->_affiliate_url( $author );
		}

		private function _affiliate_url( $url ) : string {
			if ( $this->_impact['id'] > 0 ) {
				return sprintf( 'https://1.envato.market/c/%s/%s/%s?u=%s',
					$this->_impact['id'],
					$this->_impact['ad'],
					$this->_impact['affiliate'],
					urlencode( $url ) );
			} else if ( $this->_referrer != '' ) {
				return $url . '?ref=' . $this->_referrer;
			} else {
				return $url;
			}
		}
	}
}

if ( ! class_exists( 'smart_envato_api_item' ) ) {
	/**
	 * @property $rating
	 * @property $rating_count
	 * @property $price_cents
	 * @property $number_of_sales
	 * @property $author_username
	 * @property $site
	 * @property $name
	 * @property $published_at
	 * @property $updated_at
	 * @property $classification
	 * @property $attributes
	 * @property $author_url
	 * @property $url
	 */
	class smart_envato_api_item {
		private $_data = array();

		private $_referrer;
		private $_impact = array(
			'affiliate' => 4415,
			'id'        => 0,
			'ad'        => 0,
		);
		private $_attributes = array();

		public $slug = '';
		public $preview = '';

		public $uploaded_on = '';
		public $last_update = '';
		public $date_new = '';
		public $date_updated = '';
		public $cost = '';
		public $sales = '';
		public $user = '';
		public $item = '';
		public $market = '';
		public $thumbnail = '';
		public $live_preview_url = '';
		public $video_preview_url = '';
		public $category = '';
		public $categories = '';

		public function __construct( $item, $referrer = '', $core = null, $impact = array() ) {
			if ( is_array( $item ) || is_object( $item ) ) {
				$this->_data = (array) $item;
			}

			if ( is_object( $this->rating ) ) {
				$this->rating_count = $this->rating->count;
				$this->rating       = $this->rating->rating;
			}

			$this->cost        = $this->price_cents / 100;
			$this->sales       = $this->number_of_sales;
			$this->user        = $this->author_username;
			$this->market      = substr( $this->site, 0, - 4 );
			$this->item        = $this->name;
			$this->uploaded_on = $this->published_at;
			$this->last_update = $this->updated_at;

			if ( isset( $this->url ) ) {
				$path       = wp_parse_url( $this->url, PHP_URL_PATH );
				$path_parts = explode( '/', $path );
				$this->slug = $path_parts[2];
			}

			$this->category = $this->classification;
			if ( isset( $this->category ) && ! is_null( $core ) ) {
				$this->categories = $core->data()->category_name( $this->market, $this->category );
			}

			if ( isset( $this->previews->icon_preview ) ) {
				$this->thumbnail = $this->previews->icon_preview->icon_url;
			} else if ( isset( $this->previews->icon_with_thumbnail_preview ) ) {
				$this->thumbnail = $this->previews->icon_with_thumbnail_preview->icon_url;
			} else if ( isset( $this->previews->icon_with_audio_preview ) ) {
				$this->thumbnail = $this->previews->icon_with_audio_preview->icon_url;
			} else if ( isset( $this->previews->icon_with_square_preview ) ) {
				$this->thumbnail = $this->previews->icon_with_square_preview->icon_url;
			} else if ( isset( $this->previews->icon_with_landscape_preview ) ) {
				$this->thumbnail = $this->previews->icon_with_landscape_preview->icon_url;
			} else if ( isset( $this->previews->icon_with_video_preview ) ) {
				$this->thumbnail = $this->previews->icon_with_video_preview->icon_url;
			} else if ( isset( $this->previews->thumbnail_preview ) ) {
				$this->thumbnail = $this->previews->thumbnail_preview->small_url;
			}

			if ( isset( $this->previews->landscape_preview ) ) {
				$this->live_preview_url = $this->previews->landscape_preview->landscape_url;
			} else if ( isset( $this->previews->icon_with_thumbnail_preview ) ) {
				$this->live_preview_url = $this->previews->icon_with_thumbnail_preview->thumbnail_url;
			} else if ( isset( $this->previews->icon_with_landscape_preview ) ) {
				$this->live_preview_url = $this->previews->icon_with_landscape_preview->landscape_url;
			} else if ( isset( $this->previews->icon_with_square_preview ) ) {
				$this->live_preview_url = $this->previews->icon_with_square_preview->square_url;
			} else if ( isset( $this->previews->icon_with_video_preview ) ) {
				$this->live_preview_url = $this->previews->icon_with_video_preview->landscape_url;
			} else if ( isset( $this->previews->thumbnail_preview ) ) {
				$this->live_preview_url = $this->previews->thumbnail_preview->large_url;
			}

			if ( isset( $this->previews->icon_with_video_preview ) ) {
				$this->video_preview_url = $this->previews->icon_with_video_preview->video_url;
			}

			if ( isset( $this->live_preview_url ) ) {
				$this->preview = $this->live_preview_url;
			}

			$this->_referrer = $referrer;

			$this->_impact['affiliate'] = $impact['affiliate'] ?? 4415;
			$this->_impact['id']        = $impact['id'] ?? 0;
			$this->_impact['ad']        = $impact['ad'] ?? 0;

			foreach ( $this->attributes as $group ) {
				$this->_attributes[ $group->name ] = $group;
			}
		}

		public function __get( $name ) {
			return $this->_data[ $name ] ?? '';
		}

		public function __set( $name, $value ) {
			$this->_data[ $name ] = $value;
		}

		function __isset( $name ) {
			return isset( $this->_data[ $name ] );
		}

		public function url() : string {
			return $this->_affiliate_url( $this->url );
		}

		public function author_url() : string {
			return $this->_affiliate_url( $this->author_url );
		}

		public function author_portfolio_url() : string {
			return $this->_affiliate_url( trailingslashit( $this->author_url ) . 'portfolio' );
		}

		public function preview_url() {
			if ( isset( $this->previews->live_site ) ) {
				$preview = 'https://' . $this->site . '/item/' . $this->slug . '/full_screen_preview/' . $this->id;

				return $this->_affiliate_url( $preview );
			} else {
				return false;
			}
		}

		public function screenshots_url() : string {
			$preview = 'https://' . $this->site . '/item/' . $this->slug . '/screenshots/' . $this->id;

			return $this->_affiliate_url( $preview );
		}

		public function comments_url() : string {
			return $this->_affiliate_url( trailingslashit( $this->url ) . 'comments' );
		}

		public function get_attribute( $name ) {
			if ( isset( $this->_attributes[ $name ] ) ) {
				return $this->_attributes[ $name ];
			}

			return null;
		}

		public function title() : string {
			return $this->item;
		}

		public function title_formatted() : string {
			$parts = explode( ' - ', $this->item, 2 );

			return $parts[0] . ( isset( $parts[1] ) ? '<span>' . $parts[1] . '</span>' : '' );
		}

		private function _affiliate_url( $url ) : string {
			if ( $this->_impact['id'] > 0 ) {
				return sprintf( 'https://1.envato.market/c/%s/%s/%s?u=%s',
					$this->_impact['id'],
					$this->_impact['ad'],
					$this->_impact['affiliate'],
					urlencode( $url ) );
			} else if ( $this->_referrer != '' ) {
				return $url . '?ref=' . $this->_referrer;
			} else {
				return $url;
			}
		}
	}
}
